package nemosofts.notes.app.utils.helper;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import androidx.annotation.NonNull;

import java.util.Locale;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import nemosofts.notes.app.BuildConfig;
import nemosofts.notes.app.R;
import nemosofts.notes.app.item.Cat;
import nemosofts.notes.app.item.NoteHistory;
import nemosofts.notes.app.item.Note;
import nemosofts.notes.app.utils.ApplicationUtil;

public class DBHelper extends SQLiteOpenHelper {

    private static final String TAG = "DBHelper";
    private static final int DATABASE_VERSION = 4;
    private static final String DATABASE_NAME = BuildConfig.APPLICATION_ID + "_" + "note_app.db";
    private final SQLiteDatabase db;
    private final Context context;

    private static final String TAG_ID = "id";
    private static final String TAG_PLAYLIST_ID = "id";
    private static final String TAG_PLAYLIST_NAME = "name";

    private static final String TABLE_PLAYLIST = "playlist";
    private static final String TABLE_NOTES = "notes";
    private static final String TABLE_DELETED_NOTES = "deleted_notes";

    private static final String TAG_NOTE_ID = "id";
    private static final String TAG_NOTE_TITLE = "title";
    private static final String TAG_NOTE_DATE_TIME = "date_time";
    private static final String TAG_NOTE_SUBTITLE = "subtitle";
    private static final String TAG_NOTE_TEXT = "note_text";
    private static final String TAG_NOTE_IMAGE_PATH = "image_path";
    private static final String TAG_NOTE_COLOR = "color";
    private static final String TAG_NOTE_WEB_LINK = "web_link";
    private static final String TAG_NOTE_CAT_ID = "catId";
    private static final String TAG_NOTE_REMINDER_TIME = "reminder_time";
    private static final String TAG_NOTE_REMINDER_ACTIVE = "reminder_active";
    private static final String TAG_NOTE_AUDIO_PATH = "audio_path";
    private static final String TAG_NOTE_AUDIO_DURATION = "audio_duration";
    private static final String TAG_NOTE_LAST_OPENED = "last_opened";
    private static final String TAG_NOTE_LAST_UPDATED = "last_updated";

    private static final String TABLE_NOTE_HISTORY = "note_history";
    private static final String TAG_HISTORY_ID = "id";
    private static final String TAG_HISTORY_NOTE_ID = "note_id";
    private static final String TAG_HISTORY_ACTION = "action";
    private static final String TAG_HISTORY_DETAIL = "detail";
    private static final String TAG_HISTORY_TS = "timestamp";

    private final String[] columnsPlaylist = new String[]{TAG_PLAYLIST_ID, TAG_PLAYLIST_NAME};

    private static final String CREATE_TABLE_PLAYLIST = "CREATE TABLE IF NOT EXISTS " + TABLE_PLAYLIST + " (" +
            TAG_PLAYLIST_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
            TAG_PLAYLIST_NAME + " TEXT);";

    private static final String CREATE_TABLE_NOTES = "CREATE TABLE IF NOT EXISTS " + TABLE_NOTES + " (" +
            TAG_NOTE_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
            TAG_NOTE_TITLE + " TEXT, " +
            TAG_NOTE_DATE_TIME + " TEXT, " +
            TAG_NOTE_SUBTITLE + " TEXT, " +
            TAG_NOTE_TEXT + " TEXT, " +
            TAG_NOTE_IMAGE_PATH + " TEXT, " +
            TAG_NOTE_COLOR + " TEXT, " +
            TAG_NOTE_WEB_LINK + " TEXT, " +
            TAG_NOTE_CAT_ID + " TEXT, " +
            TAG_NOTE_REMINDER_TIME + " INTEGER DEFAULT 0, " +
            TAG_NOTE_REMINDER_ACTIVE + " INTEGER DEFAULT 0, " +
            TAG_NOTE_AUDIO_PATH + " TEXT, " +
            TAG_NOTE_AUDIO_DURATION + " INTEGER DEFAULT 0, " +
            TAG_NOTE_LAST_OPENED + " INTEGER DEFAULT 0, " +
            TAG_NOTE_LAST_UPDATED + " INTEGER DEFAULT 0);";

    private static final String CREATE_TABLE_DELETED_NOTES = "CREATE TABLE IF NOT EXISTS " + TABLE_DELETED_NOTES + " (" +
            TAG_NOTE_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
            TAG_NOTE_TITLE + " TEXT, " +
            TAG_NOTE_DATE_TIME + " TEXT, " +
            TAG_NOTE_SUBTITLE + " TEXT, " +
            TAG_NOTE_TEXT + " TEXT, " +
            TAG_NOTE_IMAGE_PATH + " TEXT, " +
            TAG_NOTE_COLOR + " TEXT, " +
            TAG_NOTE_WEB_LINK + " TEXT, " +
            TAG_NOTE_CAT_ID + " TEXT, " +
            TAG_NOTE_REMINDER_TIME + " INTEGER DEFAULT 0, " +
            TAG_NOTE_REMINDER_ACTIVE + " INTEGER DEFAULT 0, " +
            TAG_NOTE_AUDIO_PATH + " TEXT, " +
            TAG_NOTE_AUDIO_DURATION + " INTEGER DEFAULT 0, " +
            TAG_NOTE_LAST_OPENED + " INTEGER DEFAULT 0, " +
            TAG_NOTE_LAST_UPDATED + " INTEGER DEFAULT 0);";

    private static final String CREATE_TABLE_NOTE_HISTORY = "CREATE TABLE IF NOT EXISTS " + TABLE_NOTE_HISTORY + " (" +
            TAG_HISTORY_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
            TAG_HISTORY_NOTE_ID + " INTEGER, " +
            TAG_HISTORY_ACTION + " TEXT, " +
            TAG_HISTORY_DETAIL + " TEXT, " +
            TAG_HISTORY_TS + " INTEGER DEFAULT 0);";

    public DBHelper(Context context) {
        super(context, DATABASE_NAME, null, DATABASE_VERSION);
        this.context = context;
        db = getWritableDatabase();
    }

    public static String getDatabaseFileName() {
        return DATABASE_NAME;
    }

    public static File getDatabaseFile(@NonNull Context context) {
        return context.getDatabasePath(DATABASE_NAME);
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        try {
            db.execSQL(CREATE_TABLE_PLAYLIST);
            db.execSQL(CREATE_TABLE_NOTES);
            db.execSQL(CREATE_TABLE_DELETED_NOTES);
            db.execSQL(CREATE_TABLE_NOTE_HISTORY);
            addPlayListMyPlay(db, context.getString(R.string.my_categories));
        } catch (Exception e) {
            ApplicationUtil.log(TAG, "Error creating table", e);
        }
    }

    // Upgrade -------------------------------------------------------------------------------------
    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        if (oldVersion < 2) {
            addReminderColumns(db, TABLE_NOTES);
            addReminderColumns(db, TABLE_DELETED_NOTES);
        }
        if (oldVersion < 3) {
            addAudioColumns(db, TABLE_NOTES);
            addAudioColumns(db, TABLE_DELETED_NOTES);
        }
        if (oldVersion < 4) {
            addActivityColumns(db, TABLE_NOTES);
            addActivityColumns(db, TABLE_DELETED_NOTES);
            try {
                db.execSQL(CREATE_TABLE_NOTE_HISTORY);
            } catch (Exception ignored) {
            }
        }
    }

    private void addReminderColumns(SQLiteDatabase db, String tableName) {
        try {
            db.execSQL("ALTER TABLE " + tableName + " ADD COLUMN " + TAG_NOTE_REMINDER_TIME + " INTEGER DEFAULT 0");
        } catch (Exception ignored) {
        }
        try {
            db.execSQL("ALTER TABLE " + tableName + " ADD COLUMN " + TAG_NOTE_REMINDER_ACTIVE + " INTEGER DEFAULT 0");
        } catch (Exception ignored) {
        }
    }

    private void addAudioColumns(SQLiteDatabase db, String tableName) {
        try {
            db.execSQL("ALTER TABLE " + tableName + " ADD COLUMN " + TAG_NOTE_AUDIO_PATH + " TEXT");
        } catch (Exception ignored) {
        }
        try {
            db.execSQL("ALTER TABLE " + tableName + " ADD COLUMN " + TAG_NOTE_AUDIO_DURATION + " INTEGER DEFAULT 0");
        } catch (Exception ignored) {
        }
    }

    private void addActivityColumns(SQLiteDatabase db, String tableName) {
        try {
            db.execSQL("ALTER TABLE " + tableName + " ADD COLUMN " + TAG_NOTE_LAST_OPENED + " INTEGER DEFAULT 0");
        } catch (Exception ignored) {
        }
        try {
            db.execSQL("ALTER TABLE " + tableName + " ADD COLUMN " + TAG_NOTE_LAST_UPDATED + " INTEGER DEFAULT 0");
        } catch (Exception ignored) {
        }
    }

    public void addPlayListMyPlay(SQLiteDatabase db, String playlist) {
        if (db != null) {
            try (Cursor cursor = db.query(TABLE_PLAYLIST, new String[]{TAG_PLAYLIST_ID},
                    null, null, null, null, null)) {
                if (cursor.moveToFirst()) {
                    return;
                }
            }
            ContentValues contentValues = new ContentValues();
            contentValues.put(TAG_PLAYLIST_NAME, playlist);
            db.insert(TABLE_PLAYLIST, null, contentValues);
        }
    }

    public List<Cat> addPlayList(String playlist) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(TAG_PLAYLIST_NAME, playlist);
        db.insert(TABLE_PLAYLIST, null, contentValues);
        return loadPlayList();
    }

    @SuppressLint("Range")
    public List<Cat> loadPlayList() {
        ArrayList<Cat> arrayList = new ArrayList<>();
        try {
            Cursor cursor = db.query(TABLE_PLAYLIST, columnsPlaylist, null, null, null, null, TAG_PLAYLIST_NAME + " ASC");
            if (cursor.moveToFirst()) {
                for (int i = 0; i < cursor.getCount(); i++) {
                    String id = cursor.getString(cursor.getColumnIndex(TAG_PLAYLIST_ID));
                    String name = cursor.getString(cursor.getColumnIndex(TAG_PLAYLIST_NAME));

                    Cat objItem = new Cat(id, name);
                    arrayList.add(objItem);

                    cursor.moveToNext();
                }
                cursor.close();
            }
        } catch (Exception e) {
            ApplicationUtil.log("loadPlayList", e.getMessage(), e);
        }
        return arrayList;
    }

    public void removePlayList(String pid) {
        db.delete(TABLE_PLAYLIST, TAG_ID + "=" + pid, null);
    }

    public void updatePlayList(String id, String name) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(TAG_PLAYLIST_NAME, name);
        db.update(TABLE_PLAYLIST, contentValues, TAG_ID + "=" + id, null);
    }

    public long saveNote(Note note) {
        ContentValues contentValues = createNoteContentValues(note);
        if (note.getId() > 0) {
            int rows = db.update(TABLE_NOTES, contentValues, TAG_NOTE_ID + "=?", new String[]{String.valueOf(note.getId())});
            if (rows == 0) {
                long newId = db.insert(TABLE_NOTES, null, contentValues);
                if (newId != -1) {
                    note.setId((int) newId);
                }
                return newId;
            }
            return note.getId();
        }
        long id = db.insert(TABLE_NOTES, null, contentValues);
        if (id != -1) {
            note.setId((int) id);
        }
        return id;
    }

    public void deleteNote(int id) {
        db.delete(TABLE_NOTES, TAG_NOTE_ID + "=?", new String[]{String.valueOf(id)});
    }

    public List<Note> getAllNotes() {
        return getNotesInternal(TABLE_NOTES, null, null);
    }

    public List<Note> getNotesByCategory(String catId) {
        return getNotesInternal(TABLE_NOTES, TAG_NOTE_CAT_ID + "=?", new String[]{catId});
    }

    public void addDeletedNote(Note note) {
        ContentValues contentValues = createNoteContentValues(note);
        if (note.getId() > 0) {
            contentValues.put(TAG_NOTE_ID, note.getId());
            db.insertWithOnConflict(TABLE_DELETED_NOTES, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);
            return;
        }
        db.insert(TABLE_DELETED_NOTES, null, contentValues);
    }

    public void deleteDeletedNote(int id) {
        db.delete(TABLE_DELETED_NOTES, TAG_NOTE_ID + "=?", new String[]{String.valueOf(id)});
    }

    public List<Note> getDeletedNotes() {
        return getNotesInternal(TABLE_DELETED_NOTES, null, null);
    }

    @NonNull
    private ContentValues createNoteContentValues(@NonNull Note note) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(TAG_NOTE_TITLE, note.getTitle());
        contentValues.put(TAG_NOTE_DATE_TIME, note.getDateTime());
        contentValues.put(TAG_NOTE_SUBTITLE, note.getSubtitle());
        contentValues.put(TAG_NOTE_TEXT, note.getNoteText());
        contentValues.put(TAG_NOTE_IMAGE_PATH, note.getImagePath());
        contentValues.put(TAG_NOTE_COLOR, note.getColor());
        contentValues.put(TAG_NOTE_WEB_LINK, note.getWebLink());
        contentValues.put(TAG_NOTE_CAT_ID, note.getCatId());
        contentValues.put(TAG_NOTE_REMINDER_TIME, note.getReminderTime());
        contentValues.put(TAG_NOTE_REMINDER_ACTIVE, note.isReminderEnabled() ? 1 : 0);
        contentValues.put(TAG_NOTE_AUDIO_PATH, note.getAudioPath());
        contentValues.put(TAG_NOTE_AUDIO_DURATION, note.getAudioDuration());
        contentValues.put(TAG_NOTE_LAST_OPENED, note.getLastOpened());
        contentValues.put(TAG_NOTE_LAST_UPDATED, note.getLastUpdated());
        return contentValues;
    }

    @NonNull
    @SuppressLint("Range")
    private List<Note> getNotesInternal(String table, String selection, String[] selectionArgs) {
        List<Note> notes = new ArrayList<>();
        try (Cursor cursor = db.query(table, null, selection, selectionArgs, null,
                null, TAG_NOTE_ID + " DESC")) {
            if (cursor.moveToFirst()) {
                for (int i = 0; i < cursor.getCount(); i++) {
                    Note note = new Note();
                    note.setId(cursor.getInt(cursor.getColumnIndex(TAG_NOTE_ID)));
                    note.setTitle(cursor.getString(cursor.getColumnIndex(TAG_NOTE_TITLE)));
                    note.setDateTime(cursor.getString(cursor.getColumnIndex(TAG_NOTE_DATE_TIME)));
                    note.setSubtitle(cursor.getString(cursor.getColumnIndex(TAG_NOTE_SUBTITLE)));
                    note.setNoteText(cursor.getString(cursor.getColumnIndex(TAG_NOTE_TEXT)));
                    note.setImagePath(cursor.getString(cursor.getColumnIndex(TAG_NOTE_IMAGE_PATH)));
                    note.setColor(cursor.getString(cursor.getColumnIndex(TAG_NOTE_COLOR)));
                    note.setWebLink(cursor.getString(cursor.getColumnIndex(TAG_NOTE_WEB_LINK)));
                    note.setCatId(cursor.getString(cursor.getColumnIndex(TAG_NOTE_CAT_ID)));
                    int reminderTimeIndex = cursor.getColumnIndex(TAG_NOTE_REMINDER_TIME);
                    int reminderActiveIndex = cursor.getColumnIndex(TAG_NOTE_REMINDER_ACTIVE);
                    if (reminderTimeIndex >= 0) {
                        note.setReminderTime(cursor.getLong(reminderTimeIndex));
                    }
                    note.setReminderEnabled(reminderActiveIndex >= 0 && cursor.getInt(reminderActiveIndex) == 1);
                    int audioPathIndex = cursor.getColumnIndex(TAG_NOTE_AUDIO_PATH);
                    int audioDurationIndex = cursor.getColumnIndex(TAG_NOTE_AUDIO_DURATION);
                    if (audioPathIndex >= 0) {
                        note.setAudioPath(cursor.getString(audioPathIndex));
                    }
                    if (audioDurationIndex >= 0) {
                        note.setAudioDuration(cursor.getLong(audioDurationIndex));
                    }
                    int lastOpenedIndex = cursor.getColumnIndex(TAG_NOTE_LAST_OPENED);
                    int lastUpdatedIndex = cursor.getColumnIndex(TAG_NOTE_LAST_UPDATED);
                    if (lastOpenedIndex >= 0) {
                        note.setLastOpened(cursor.getLong(lastOpenedIndex));
                    }
                    if (lastUpdatedIndex >= 0) {
                        note.setLastUpdated(cursor.getLong(lastUpdatedIndex));
                    }
                    notes.add(note);
                    cursor.moveToNext();
                }
            }
        }
        return notes;
    }

    public void updateLastOpened(int noteId, long timestamp) {
        ContentValues cv = new ContentValues();
        cv.put(TAG_NOTE_LAST_OPENED, timestamp);
        db.update(TABLE_NOTES, cv, TAG_NOTE_ID + "=?", new String[]{String.valueOf(noteId)});
    }

    public void addHistory(int noteId, String action, String detail, long timestamp) {
        ContentValues cv = new ContentValues();
        cv.put(TAG_HISTORY_NOTE_ID, noteId);
        cv.put(TAG_HISTORY_ACTION, action);
        cv.put(TAG_HISTORY_DETAIL, detail);
        cv.put(TAG_HISTORY_TS, timestamp);
        db.insert(TABLE_NOTE_HISTORY, null, cv);
    }

    @NonNull
    @SuppressLint("Range")
    public List<NoteHistory> getHistory(int noteId) {
        List<NoteHistory> history = new ArrayList<>();
        try (Cursor cursor = db.query(TABLE_NOTE_HISTORY, null,
                TAG_HISTORY_NOTE_ID + "=?", new String[]{String.valueOf(noteId)},
                null, null, TAG_HISTORY_TS + " DESC")) {
            if (cursor.moveToFirst()) {
                do {
                    NoteHistory item = new NoteHistory();
                    item.setId(cursor.getInt(cursor.getColumnIndex(TAG_HISTORY_ID)));
                    item.setNoteId(cursor.getInt(cursor.getColumnIndex(TAG_HISTORY_NOTE_ID)));
                    item.setAction(cursor.getString(cursor.getColumnIndex(TAG_HISTORY_ACTION)));
                    item.setDetail(cursor.getString(cursor.getColumnIndex(TAG_HISTORY_DETAIL)));
                    item.setTimestamp(cursor.getLong(cursor.getColumnIndex(TAG_HISTORY_TS)));
                    history.add(item);
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            ApplicationUtil.log(TAG, String.format(Locale.US, "History load failed for %d", noteId), e);
        }
        return history;
    }

    public Note getNoteById(int id) {
        List<Note> notes = getNotesInternal(TABLE_NOTES, TAG_NOTE_ID + "=?", new String[]{String.valueOf(id)});
        if (notes.isEmpty()) {
            return null;
        }
        return notes.get(0);
    }

    @Override
    public synchronized void close () {
        if (db != null && db.isOpen()) {
            db.close();
            super.close();
        }
    }
}
